package ir.iiscenter.shub;

import android.annotation.TargetApi;
import android.content.Context;
import android.graphics.Bitmap;
import android.net.http.SslError;
import android.os.Build;
import android.webkit.SslErrorHandler;
import android.webkit.WebResourceError;
import android.webkit.WebResourceRequest;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

/**
 * Created by taesiri on 10/23/16.
 */

class ShubViewClient extends WebViewClient {

    private MainActivity mainActivity;
    private List<String> urlHistory = new ArrayList<>();

    ShubViewClient(Context context) {
        if (context != null) {
            if (context instanceof MainActivity) {
                mainActivity = (MainActivity) context;
            }
        }
    }

    // Begin shouldOverrideUrlLoading
    @TargetApi(Build.VERSION_CODES.N)
    @Override
    public boolean shouldOverrideUrlLoading(WebView view, WebResourceRequest url) {
        // return if url is null (can happen if clicking refresh when there is no page loaded)
        return url != null && shouldOverrideUrlLoading(view, url.getUrl().toString());
    }

    @SuppressWarnings("deprecation")
    @Override
    public boolean shouldOverrideUrlLoading(WebView view, String url) {
        // return if url is null (can happen if clicking refresh when there is no page loaded)
        if (url == null) return false;

        if(mainActivity!=null) {
            mainActivity.loadUrl(url, view);
            return true;
        }
        return false;
    }
    // End shouldOverrideUrlLoading

    @Override
    public void onPageFinished(WebView view, String url) {
        super.onPageFinished(view, url);
        if(mainActivity!=null) {
            boolean complete = true;
            Information inf = Config.getInstance(mainActivity).getInformation();
            if(inf!=null && inf.versionInformation.isCurrentDeprecated()){
                complete = false;
            }
            mainActivity.loadingPageFinished(complete);
            mainActivity.putValidUrlToHistory(url);
            //store cookie inside shared preferences
            CookieUtils.storeCookiesInSharedPreferences(mainActivity);
        }
    }

    @Override
    public void onPageStarted(WebView webView, String url, Bitmap favicon) {
        super.onPageStarted(webView, url, favicon);
        if(mainActivity!=null) {
            try {
                // HACK: onPageFinished() Doesn't get called after loadDataWithBaseURL; but onPageStarted() gets called
                //
                // The immediate solution (but also not good one) is not showing Loading Indicator
                //
                URL targetURL = new URL(url);
                if(targetURL.getProtocol().equals("file")) {
                    // Do nothing
                } else {
                    mainActivity.pageStartToLoad();
                    mainActivity.fetchAllMenuItems();
                }
            } catch (MalformedURLException e) {
                e.printStackTrace();
            }
            addUrlToHistory(url);
        }
    }

    @Override
    public void onReceivedSslError(WebView view, SslErrorHandler handler, SslError error) {
        handler.proceed();// Ignore SSL certificate errors
    }

    // Begin Handling onReceivedError
    // Apparently non-deprecated version of onReceivedError method does not detect all errors.
    // This technique was suggested in http://stackoverflow.com/a/33419123/846210 as a workaround
    @SuppressWarnings("deprecation")
    @Override
    public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {

        // Error Handling
        if(mainActivity == null){
            super.onReceivedError(view, errorCode, description, failingUrl);
            return;
        }

        if (urlHistory.contains(failingUrl)) {

            StringBuilder buf = new StringBuilder();
            InputStream htmlFile;

            try {
                // Reading HTML Template from resources
                htmlFile = mainActivity.getAssets().open("custom_messages/error_message.html");
                BufferedReader in = new BufferedReader(new InputStreamReader(htmlFile, "UTF-8"));

                String currentLine;

                while ((currentLine=in.readLine()) != null) {
                    buf.append(currentLine);
                }

                in.close();

                String htmlContent = buf.toString();

                // Replacing placeholders in HTML file with appropriate values
                String[][] replacements;

                if (errorCode == ERROR_TIMEOUT || errorCode == ERROR_CONNECT || errorCode == ERROR_HOST_LOOKUP  || errorCode == ERROR_IO   ) {
                    replacements = new String[][] {
                            {"@@ERROR_MESSAGE@@", mainActivity.getResources().getString( R.string.ERROR_CONNECTION)},
                            {"@@RELOAD_TEXT@@", mainActivity.getResources().getString( R.string.try_again)},
                            {"@@TARGET_URL@@", Config.getInstance(view.getContext()).getHomeUrl()}
                    };
                } else if (errorCode == ERROR_FAILED_SSL_HANDSHAKE) {
                    replacements = new String[][] {
                            {"@@ERROR_MESSAGE@@", mainActivity.getResources().getString( R.string.ERROR_FAILED_SSL_HANDSHAKE)},
                            {"@@RELOAD_TEXT@@", mainActivity.getResources().getString( R.string.try_again)},
                            {"@@TARGET_URL@@", Config.getInstance(view.getContext()).getHomeUrl()}
                    };

                } else if (errorCode == ERROR_AUTHENTICATION) {
                    replacements = new String[][] {
                            {"@@ERROR_MESSAGE@@",  mainActivity.getResources().getString( R.string.ERROR_AUTHENTICATION)},
                            {"@@RELOAD_TEXT@@", mainActivity.getResources().getString( R.string.try_again)},
                            {"@@TARGET_URL@@", Config.getInstance(view.getContext()).getHomeUrl()}
                    };

                } else if (errorCode == ERROR_BAD_URL) {
                    replacements = new String[][] {
                            {"@@ERROR_MESSAGE@@",  mainActivity.getResources().getString( R.string.ERROR_BAD_URL)},
                            {"@@RELOAD_TEXT@@", mainActivity.getResources().getString( R.string.try_again)},
                            {"@@TARGET_URL@@", Config.getInstance(view.getContext()).getHomeUrl()}
                    };


                } else if (errorCode == ERROR_UNSUPPORTED_SCHEME) {
                    replacements = new String[][] {
                            {"@@ERROR_MESSAGE@@", mainActivity.getResources().getString( R.string.ERROR_UNSUPPORTED_SCHEME)},
                            {"@@RELOAD_TEXT@@", mainActivity.getResources().getString( R.string.try_again)},
                            {"@@TARGET_URL@@", Config.getInstance(view.getContext()).getHomeUrl()}
                    };


                } else if (errorCode == ERROR_UNKNOWN) {
                    replacements = new String[][] {
                            {"@@ERROR_MESSAGE@@", mainActivity.getResources().getString( R.string.ERROR_UNKNOWN)},
                            {"@@RELOAD_TEXT@@", mainActivity.getResources().getString( R.string.try_again)},
                            {"@@TARGET_URL@@", Config.getInstance(view.getContext()).getHomeUrl()}
                    };

                } else {
                    // Showing ERROR_CONNECTION as a default error message
                    replacements = new String[][] {
                            {"@@ERROR_MESSAGE@@", mainActivity.getResources().getString( R.string.ERROR_CONNECTION)},
                            {"@@RELOAD_TEXT@@", mainActivity.getResources().getString( R.string.try_again)},
                            {"@@TARGET_URL@@", Config.getInstance(view.getContext()).getHomeUrl()}
                    };
                }

                if ( replacements != null) {
                    for(String[] replacement: replacements) {
                        htmlContent = htmlContent.replace(replacement[0], replacement[1]);
                    }


                    //TODO: Apparently loadDataWithBaseURL doesn't work properly on JELLY BEAN.
                    // HACK Using loadData in API levels older than KITKAT as a temporarily workaround
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                        // Need to set BaseURL for Same Origin Policy
                        view.loadDataWithBaseURL("file:///android_asset/custom_messages/", htmlContent , "text/html; charset=utf-8", "utf-8", null );
                    } else {
                        view.loadData(htmlContent,  "text/html; charset=utf-8", "utf-8");
                    }

                    // WARNING onPageFinished() Doesn't get called after loadDataWithBaseURL; but onPageStarted() gets called
                } else {
                    //Log.d(TAG, "replacements is null");
                }

            } catch (IOException e) {
                e.printStackTrace();
            }
        } else {
            // The failingUrl belongs to a Resource
            // Do Nothing Here
        }
    }

    // For more recent
    @TargetApi(android.os.Build.VERSION_CODES.M)
    @Override
    public void onReceivedError(WebView view, WebResourceRequest req, WebResourceError rErr) {
        if(mainActivity == null){
            super.onReceivedError(view, req, rErr);
            return;
        }

        // Redirect to deprecated method, so we can use it in all SDK versions
        onReceivedError(view, rErr.getErrorCode(), rErr.getDescription().toString(), req.getUrl().toString());
    }
    // End Handling onReceivedError

    private void addUrlToHistory(String url){
        if(!urlHistory.contains(url)){
            urlHistory.add(url);
        }
    }
}
