<?php
/**
 * Copyright (c) 2016, Yaser Alimardany
 * All rights reserved.
 */

/**
 *
 *
 * @author Yaser Alimardany <yaser.alimardany@gmail.com>
 * @package ow_plugins.iismobilesupport.bol
 * @since 1.0
 */
class IISMOBILESUPPORT_BOL_WebServiceGroup
{
    private static $classInstance;

    public static function getInstance()
    {
        if ( self::$classInstance === null )
        {
            self::$classInstance = new self();
        }

        return self::$classInstance;
    }

    private function __construct()
    {
    }

    public function leave(){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);
        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        if ( !OW::getUser()->isAuthorized('groups', 'view') )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $userId = OW::getUser()->getId();

        if (!isset($_GET['groupId']))
        {
            return array('valid' => false, 'message' => 'input_error');
        }

        $groupId = $_GET['groupId'];
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'input_error');
        }

        $userInvitedBefore = GROUPS_BOL_Service::getInstance()->findUser($groupId, $userId);
        if($userInvitedBefore != null) {
            if($group->userId == $userId){
                GROUPS_BOL_Service::getInstance()->deleteGroup($groupId);
            }else{
                $userIds = array($userId);
                $deleteUser = GROUPS_BOL_Service::getInstance()->deleteUser($groupId, $userIds);
                if (!$deleteUser) {
                    return array('valid' => true, 'leavable' => false);
                }
            }
            $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
            if($group == null){
                return array('valid' => true, 'leavable' => true, 'group' => null);
            }
            $groupData = $this->getGroupInformation($group);
            return array('valid' => true, 'leavable' => true, 'group' => $groupData);
        }
        return array('valid' => false, 'message' => 'authorization_error');
    }

    public function getUnreadGroupsCount() {
        if (!OW::getUser()->isAuthenticated()) {
            return 0;
        }

        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);
        if(!$pluginActive){
            return 0;
        }
        return (int) GROUPS_BOL_Service::getInstance()->getUnreadGroupsCountForUser(OW::getUser()->getId());
    }

    public function deleteGroup(){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);
        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        if ( !OW::getUser()->isAuthenticated() || !OW::getUser()->isAuthorized('groups', 'view') )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        if (!isset($_GET['groupId']))
        {
            return array('valid' => false, 'message' => 'input_error');
        }

        $groupId = $_GET['groupId'];
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'input_error');
        }

        $isOwner = OW::getUser()->getId() == $group->userId;
        $isModerator = OW::getUser()->isAuthorized('groups');
        $isAdmin = OW::getUser()->isAdmin();

        $isManager=false;
        $eventIisGroupsPlusManager = new OW_Event('iisgroupsplus.check.user.manager.status', array('groupId' => $group->id));
        OW::getEventManager()->trigger($eventIisGroupsPlusManager);
        if(isset($eventIisGroupsPlusManager->getData()['isUserManager'])){
            $isManager=$eventIisGroupsPlusManager->getData()['isUserManager'];
        }
        if ( !$isOwner && !$isModerator  && !$isManager && !$isAdmin )
        {
            return array('valid' => false, 'message' => 'input_error');
        }

        GROUPS_BOL_Service::getInstance()->deleteGroup($group->id);

        return array('valid' => true, 'leavable' => true, 'groupId' => $group->id);
    }

    public function acceptInvite(){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);
        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        if ( !OW::getUser()->isAuthenticated() || !OW::getUser()->isAuthorized('groups', 'view') )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $accepterUserId = OW::getUser()->getId();

        if (!isset($_GET['groupId']))
        {
            return array('valid' => false, 'message' => 'input_error');
        }

        $groupId = $_GET['groupId'];
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'input_error');
        }

        $invite = GROUPS_BOL_Service::getInstance()->findInvite($groupId, $accepterUserId);
        if($invite != null ){
            GROUPS_BOL_Service::getInstance()->addUser($groupId, $accepterUserId);
        } else {
            return array('valid' => false, 'message' => 'authorization_error');
        }
        return array('valid' => true, 'registration' => true, 'id' => (int) $group->id, 'group' => $this->getGroupInformation($group));
    }


    public function removeUser(){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);
        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        if ( !OW::getUser()->isAuthenticated() || !OW::getUser()->isAuthorized('groups', 'view') )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $userId = null;
        $groupId = null;

        if (!isset($_GET['userId']) || !isset($_GET['groupId']))
        {
            return array('valid' => false, 'message' => 'input_error');
        }

        $userId = $_GET['userId'];
        $groupId = $_GET['groupId'];
        return $this->removeUserById($userId, $groupId);
    }

    public function removeUserById($userId, $groupId){
        if($userId == null){
            return array('valid' => false, 'message' => 'input_error');
        }

        if($groupId == null){
            return array('valid' => false, 'message' => 'input_error');
        }

        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        if($userId == OW::getUser()->getId() || $group->userId == $userId){
            return array('valid' => false, 'message' => 'admin_delete_error');
        }

        $canEdit = GROUPS_BOL_Service::getInstance()->isCurrentUserCanEdit($group);
        if($canEdit){
            $groupUser = GROUPS_BOL_Service::getInstance()->findUser($groupId, $userId);
            if($groupUser == null){
                return array('valid' => false, 'message' => 'authorization_error');
            }
            $userIds = array($userId);
            $result = GROUPS_BOL_Service::getInstance()->deleteUser($groupId, $userIds);
            if ($result == false) {
                return array('valid' => false, 'message' => 'authorization_error', 'leavable' => false);
            }
            $groupInfo = $this->getGroupInformation($group);
            return array('valid' => true, 'message' => 'deleted', 'group' => $groupInfo);
        }
        return array('valid' => false, 'message' => 'authorization_error');
    }

    public function cancelInvite(){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);
        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        if ( !OW::getUser()->isAuthenticated() || !OW::getUser()->isAuthorized('groups', 'view') )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $accepterUserId = OW::getUser()->getId();

        if (!isset($_GET['groupId']))
        {
            return array('valid' => false, 'message' => 'input_error');
        }

        $groupId = $_GET['groupId'];
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'input_error');
        }

        $invite = GROUPS_BOL_Service::getInstance()->findInvite($groupId, $accepterUserId);
        if($invite != null ){
            GROUPS_BOL_Service::getInstance()->deleteInvite($groupId, $accepterUserId);
        } else {
            return array('valid' => false, 'message' => 'authorization_error');
        }
        return array('valid' => true, 'registration' => false, 'id' => (int) $group->id );
    }

    public function inviteUser(){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);
        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        if ( !OW::getUser()->isAuthenticated() || !OW::getUser()->isAuthorized('groups', 'view') )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }


        $inviterUserId = OW::getUser()->getId();

        if ( !isset($_GET['userId']) || !isset($_GET['groupId']) )
        {
            return array('valid' => false, 'message' => 'input_error');
        }

        $userId = $_GET['userId'];
        $groupId = $_GET['groupId'];
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'input_error');
        }
        if(!OW::getUser()->isAdmin()){
            if(!GROUPS_BOL_Service::getInstance()->isCurrentUserInvite($group->id)){
                return array('valid' => false, 'message' => 'authorization_error');
            }
        }

        $eventIisGroupsPlusCheckCanSearchAll = new OW_Event('iisgroupsplus.check.can.invite.all',array('checkAccess'=>true));
        OW::getEventManager()->trigger($eventIisGroupsPlusCheckCanSearchAll);
        if(isset($eventIisGroupsPlusCheckCanSearchAll->getData()['hasAccess'])){
            $hasAccess=true;
        }
        if(isset($eventIisGroupsPlusCheckCanSearchAll->getData()['directInvite']) && $eventIisGroupsPlusCheckCanSearchAll->getData()['directInvite']==true){
            $eventIisGroupsPlusAddAutomatically = new OW_Event('iisgroupsplus.add.users.automatically',array('groupId'=>$groupId,'userIds'=>array($userId)));
            OW::getEventManager()->trigger($eventIisGroupsPlusAddAutomatically);
            return array('valid' => true, 'result_key' => 'add_automatically');
        }else {
            if (isset($hasAccess)) {
                $result = GROUPS_BOL_Service::getInstance()->inviteUser($group->id, $userId, $inviterUserId);
                return array('valid' => $result);
            }
            if(IISSecurityProvider::checkPluginActive('friends', true)) {
                $isFriends = FRIENDS_BOL_Service::getInstance()->findFriendship($userId, $inviterUserId);
                if (isset($isFriends) && $isFriends->status == 'active') {
                    $result = GROUPS_BOL_Service::getInstance()->inviteUser($group->id, $userId, $inviterUserId);
                    return array('valid' => $result);
                }
            }
        }
        return array('valid' => false, 'message' => 'input_error');
    }

    public function getInvitableUsers(){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);
        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        if ( !OW::getUser()->isAuthenticated() || !OW::getUser()->isAuthorized('groups', 'view') )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $currentUserId = OW::getUser()->getId();

        if ( !isset($_GET['groupId']) )
        {
            return array('valid' => false, 'message' => 'input_error');
        }

        $groupId = $_GET['groupId'];
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'input_error');
        }
        if(!OW::getUser()->isAdmin()){
            if(!GROUPS_BOL_Service::getInstance()->isCurrentUserInvite($group->id)){
                return array('valid' => false, 'message' => 'authorization_error');
            }
        }

        $first = 0;
        $count = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->getPageSize();
        if(isset($_GET['first'])){
            $first = (int) $_GET['first'];
        }

        $key = '';
        if(isset($_GET['key'])){
            $key = $_GET['key'];
        }

        $idList = GROUPS_BOL_Service::getInstance()->getInvitableUserIds($groupId, $currentUserId);
        $users = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->populateInvitableUserList($idList, $key, $first, $count);
        return $users;
    }

    public function getGroups($type){
        if($type != "latest"){
            return array();
        }
        $userId = null;
        if(isset($_GET['userId'])){
            $userId = $_GET['userId'];
        }else if(isset($_GET['username'])){
            $user = BOL_UserService::getInstance()->findByUsername($_GET['username']);
            if($user != null){
                $userId = $user->getId();
            }
        }

        if(OW::getUser()->isAuthenticated() && isset($_GET['my']) && $_GET['my']){
            $userId = OW::getUser()->getId();
        }

        $data = $this->getGroupsByUserId($userId, $type);
        return $data;
    }

    public function getGroupsByUserId($userId = null, $type = 'latest'){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);

        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        $guestAccess = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->checkGuestAccess();
        if(!$guestAccess){
            return array('valid' => false, 'message' => 'guest_cant_view');
        }

        if ( !OW::getUser()->isAuthorized('groups', 'view') )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        if($userId != null){
            $checkPrivacy = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->checkPrivacyAction($userId, 'view_my_groups', 'groups');
            if(!$checkPrivacy){
                return array();
            }
        }

        $groups = array();
        $first = 0;
        $count = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->getPageSize();
        if(isset($_GET['first'])){
            $first = (int) $_GET['first'];
        }

        if($userId == null){
            $groups = GROUPS_BOL_Service::getInstance()->findGroupList($type, $first, $count, true);
        }else{
            $groups = GROUPS_BOL_Service::getInstance()->findUserGroupList($userId, $first, $count);
        }

        $groupsAdditionalInfo = array(
            'checkCanView' => false,
        );
        if ($userId == null || $userId == OW::getUser()->getId()) {
            $groupsAdditionalInfo['checkUserExistInGroup'] = false;
        }
        $data = $this->getGroupsInformation($groups, 0, 2, array(), $groupsAdditionalInfo);

        return $data;
    }

    public function getGroup(){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);

        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        $guestAccess = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->checkGuestAccess();
        if(!$guestAccess){
            return array('valid' => false, 'message' => 'guest_cant_view');
        }

        if ( !OW::getUser()->isAuthorized('groups', 'view') )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $groupId = null;
        $first = 0;
        $count = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->getPageSize();
        if(isset($_GET['first'])){
            $first = (int) $_GET['first'];
        }

        if(isset($_GET['groupId'])){
            $groupId = (int) $_GET['groupId'];
        }

        if($groupId == null){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'authorization_error', 'id' => $groupId);
        }

        $cachedActionIds = array();
        if (isset($_POST['cachedActionIds'])) {
            $cachedActionIds = $_POST['cachedActionIds'];
            $cachedActionIds = explode(',', $cachedActionIds);
            if (sizeof($cachedActionIds) > 0 && $cachedActionIds[sizeof($cachedActionIds) - 1] === '') {
                unset($cachedActionIds[sizeof($cachedActionIds) - 1]);
            }
        }

        $additionalInfo = array();
        if (sizeof($cachedActionIds) > 0) {
            $cachedActions = array();
            $tempCachedActions = NEWSFEED_BOL_ActionDao::getInstance()->findByIdList($cachedActionIds);
            foreach ($tempCachedActions as $tempCachedAction) {
                $actionDataJson = null;
                if(isset($tempCachedAction->data)){
                    $actionDataJson = $tempCachedAction->data;
                }

                if($actionDataJson != null){
                    $actionDataJson = json_decode($actionDataJson);
                }

                if($actionDataJson != null && isset($actionDataJson->contextFeedId) && $actionDataJson->contextFeedId == $groupId && isset($actionDataJson->contextFeedType) && $actionDataJson->contextFeedType == 'groups'){
                    $cachedActions[] = $tempCachedAction;
                }
            }
            if (sizeof($cachedActions) > 0) {
                $additionalInfo['doPrepareActions'] = $cachedActions;
            }
        }

        $data = $this->getGroupInformation($group, $first, $count, array('users', 'files', 'posts', 'subGroups'), $additionalInfo);
        if(IISSecurityProvider::checkPluginActive('forum', true)) {
            $data['topics'] = $this->getGroupTopics($group);
        }

        OW::getEventManager()->trigger(new OW_Event('newsfeed.feed.render', array('feedType' => 'groups', 'feedId' => $groupId)));

        if (sizeof($cachedActionIds) > 0 && isset($data['posts']) && sizeof($data['posts']) > 0) {
            $deletedActionIds = array();
            $existActionIds = array();
            foreach ($data['posts'] as $post) {
                $existActionIds[] = $post['actionId'];
            }
            foreach ($cachedActionIds as $cachedActionId) {
                if (!in_array($cachedActionId, $existActionIds)) {
                    $deletedActionIds[] = $cachedActionId;
                }
            }
            if (sizeof($deletedActionIds) > 0) {
                $data['deletedPostIds'] = $deletedActionIds;
            }
        }

        $data['unreadCount'] = 0;

        return $data;
    }

    public function getGroupTopics($group) {
        if ( $group == null)
        {
            return array();
        }
        if (!GROUPS_BOL_Service::getInstance()->isCurrentUserCanView($group))
        {
            return array();
        }

        $entity = 'groups';
        $entityId = $group->id;
        $forumService = FORUM_BOL_ForumService::getInstance();
        $forumGroup = $forumService->findGroupByEntityId($entity, $entityId);
        if ( empty($forumGroup) )
        {
            return array();
        }

        $first = 0;
        $count = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->getPageSize();
        if(isset($_GET['first'])){
            $first = (int) $_GET['first'];
        }
        $page = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->getPageNumber($first);

        $topicList = $forumService->getGroupTopicList($forumGroup->getId(), $page, $count);
        $topicsData = array();
        foreach ($topicList as $topic){
            $topicsData[] = IISMOBILESUPPORT_BOL_WebServiceForum::getInstance()->preparedTopic($topic);
        }

        return $topicsData;
    }

    public function canAddTopicToGroup($groupId, $info = array()){

        $params = array('entity' => 'groups', 'entityId' => (int) $groupId, 'action' => 'add_topic', 'info' => $info);
        $event = new OW_Event('forum.check_permissions', $params);
        OW::getEventManager()->trigger($event);

        $canAdd = $event->getData();

        if (isset($info['canCreatePost'])) {
            if ($info['canCreatePost'] == false) {
                $canAdd=false;
            }
        } else {
            $channelEvent = OW::getEventManager()->trigger(new OW_Event('iisgroupsplus.on.channel.add.widget',
                array('groupId'=>$groupId)));
            $isChannelParticipant = $channelEvent->getData()['channelParticipant'];
            if(isset($isChannelParticipant) && $isChannelParticipant){
                $canAdd=false;
            }
        }

        $canCreateTopic = false;
        if(IISSecurityProvider::checkPluginActive('iisgroupsplus', true)){
            if (isset($info['groupSetting'])) {
                $groupSetting = $info['groupSetting'];
            } else {
                $groupSetting = IISGROUPSPLUS_BOL_GroupSettingDao::getInstance()->findByGroupId($groupId);
            }
            if (isset($groupSetting)){
                if($groupSetting->whoCanCreateTopic == IISGROUPSPLUS_BOL_Service::WCU_MANAGERS)
                {
                    $canCreateTopic = true;
                }
            }
        }

        if ($canCreateTopic) {
            return true;
        }

        return $canAdd;
    }

    public function canUserCreateGroup(){
        if(!IISSecurityProvider::checkPluginActive('groups', true)){
            return false;
        }

        if ( !OW::getUser()->isAuthenticated() || !OW::getUser()->isAuthorized('groups', 'create') ){
            return false;
        }

        return true;
    }

    public function canUserAccessWithEntity($entityType, $entityId){
        if(!IISSecurityProvider::checkPluginActive('groups', true)){
            return false;
        }
        $activity = IISMOBILESUPPORT_BOL_WebServiceNewsfeed::getInstance()->getCreatorActivityOfAction($entityType, $entityId);
        if($activity == null){
            return false;
        }
        $feedIdFromActivities = NEWSFEED_BOL_ActionFeedDao::getInstance()->findByActivityIds(array($activity->id));
        $group = null;
        foreach ($feedIdFromActivities as $feedFromActivity){
            if($feedFromActivity->feedType=="groups"){
                $group = GROUPS_BOL_Service::getInstance()->findGroupById($feedFromActivity->feedId);
            }
        }
        if($group == null){
            return false;
        }
        if ( !GROUPS_BOL_Service::getInstance()->isCurrentUserCanView($group) )
        {
            return false;
        }

        return true;
    }

    public function removeGroupManager(){
        $pluginActive = IISSecurityProvider::checkPluginActive('iisgroupsplus', true);
        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        $managerId = null;
        $groupId = null;
        if (isset($_POST['userId'])){
            $managerId = (int) $_POST['userId'];
        }
        if (isset($_POST['groupId'])){
            $groupId = (int) $_POST['groupId'];
        }
        if($managerId == null || $groupId == null || !OW::getUser()->isAuthenticated()) {
            return array('valid' => false, 'message' => 'input_error');
        }
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $canManage = GROUPS_BOL_Service::getInstance()->isCurrentUserCanEdit($group);
        if (!$canManage) {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $groupUser = GROUPS_BOL_Service::getInstance()->findUser($groupId, $managerId);
        if($groupUser == null){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $isManager = IISGROUPSPLUS_BOL_GroupManagersDao::getInstance()->getGroupManagerByUidAndGid($groupId, $managerId);
        if ($isManager) {
            IISGROUPSPLUS_BOL_GroupManagersDao::getInstance()->deleteGroupManagerByUidAndGid($groupId, [$managerId]);
        }

        return array('valid' => true, 'userId' => (int) $managerId, 'groupId' => (int) $groupId);
    }

    public function addGroupManager(){
        $pluginActive = IISSecurityProvider::checkPluginActive('iisgroupsplus', true);
        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        $managerId = null;
        $groupId = null;
        if (isset($_POST['userId'])){
            $managerId = (int) $_POST['userId'];
        }
        if (isset($_POST['groupId'])){
            $groupId = (int) $_POST['groupId'];
        }
        if($managerId == null || $groupId == null || !OW::getUser()->isAuthenticated()) {
            return array('valid' => false, 'message' => 'input_error');
        }
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $canManage = GROUPS_BOL_Service::getInstance()->isCurrentUserCanEdit($group);
        if (!$canManage) {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $groupUser = GROUPS_BOL_Service::getInstance()->findUser($groupId, $managerId);
        if($groupUser == null){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $isManager = IISGROUPSPLUS_BOL_GroupManagersDao::getInstance()->getGroupManagerByUidAndGid($groupId, $managerId);
        if (!$isManager) {
            IISGROUPSPLUS_BOL_GroupManagersDao::getInstance()->addUserAsManager($groupId, $managerId);
        }

        return array('valid' => true, 'userId' => (int) $managerId, 'groupId' => (int) $groupId);
    }

    public function addFile(){
        if ( !isset($_POST['groupId']) )
        {
            return array('valid' => false, 'message' => 'input_error');
        }

        $groupId = $_POST['groupId'];
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'input_error');
        }

        if(!$this->canAddFile($group)){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        if (isset($_FILES['file']) && isset($_FILES['file']['tmp_name'])) {
            $isFileClean = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->isFileClean($_FILES['file']['tmp_name']);
            if (!$isFileClean) {
                return array('valid' => false, 'message' => 'virus_detected');
            }
        }

        $resultArr = IISGROUPSPLUS_BOL_Service::getInstance()->manageAddFile($groupId, $_FILES['file']);
        if(!isset($resultArr) || !$resultArr['result']){
            return array('valid' => false, 'message' => 'authorization_error');
        }
        OW::getEventManager()->call('iisfilemanager.after_file_upload',
            array('entityType'=>'groups', 'entityId'=>$groupId, 'dto'=>$resultArr['dtoArr']['dto'], 'file' => $_FILES['file']));

        $filesList = IISGROUPSPLUS_BOL_Service::getInstance()->findFileList($group->id, 0, 1);
        $filesInformation = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->preparedFileList($group, $filesList);

        return array('valid' => true, 'files' => $filesInformation);
    }

    public function deleteFile(){
        if ( !isset($_POST['groupId']) || !isset($_POST['id']) )
        {
            return false;
        }

        $groupId = $_POST['groupId'];
        $attachmentId = $_POST['id'];
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'input_error');
        }

        $canDeleteGroupFile = $this->canDeleteFile($group);
        $attachment = BOL_AttachmentDao::getInstance()->findById($attachmentId);
        $canDeleteFile = true;
        if ($attachment->userId != OW::getUser()->getId()) {
            $canDeleteFile = false;
        }
        if(!$canDeleteFile && !$canDeleteGroupFile){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        try {
            IISGROUPSPLUS_BOL_Service::getInstance()->deleteFileForGroup($groupId, $attachmentId);
            return array('valid' => true, 'id' => (int) $attachmentId);
        }
        catch (Exception $e){
            return array('valid' => false, 'message' => 'authorization_error');
        }
    }

    public function editFile(){
        if ( !isset($_POST['groupId']) || !isset($_POST['id']) )
        {
            return false;
        }

        $groupId = $_POST['groupId'];
        $attachmentId = $_POST['id'];
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'input_error');
        }

        $canDeleteGroupFile = $this->canDeleteFile($group);
        $attachment = BOL_AttachmentDao::getInstance()->findById($attachmentId);
        $canDeleteFile = true;
        if ($attachment->userId != OW::getUser()->getId()) {
            $canDeleteFile = false;
        }
        if(!$canDeleteFile && !$canDeleteGroupFile){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        try {
            $new_name = isset($_POST['new_name'])?$_POST['new_name']:null;
            $new_parent_id = isset($_POST['new_parent_id'])?$_POST['new_parent_id']:null;
            BOL_AttachmentService::getInstance()->editAttachmentById($attachmentId, $new_name, $new_parent_id);
            return array('valid' => true, 'id' => (int) $attachmentId);
        }
        catch (Exception $e){
            return array('valid' => false, 'message' => 'authorization_error');
        }
    }

    public function addDir(){
        if ( !isset($_POST['groupId']) || !isset($_POST['name']) || !isset($_POST['parent_id']) ){
            return array('valid' => false, 'message' => 'input_error');
        }
        if( !IISSecurityProvider::checkPluginActive('iisfilemanager', true)) {
            return array('valid' => false, 'message' => 'input_error');
        }
        $groupId = $_POST['groupId'];
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'input_error');
        }
        if(!$this->canAddFile($group)){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $service = IISFILEMANAGER_BOL_Service::getInstance();
        $service->insert($_POST['name'], $_POST['parent_id'], 'directory', time(), '', true, false);
        $subFolders = $service->getSubfolders('groups', (int) $group->id);
        return array('valid' => true, 'subfolders' => $subFolders);
    }

    public function editDir(){
        if ( !isset($_POST['groupId']) || !isset($_POST['id'])){
            return array('valid' => false, 'message' => 'input_error');
        }
        if( !IISSecurityProvider::checkPluginActive('iisfilemanager', true)) {
            return array('valid' => false, 'message' => 'input_error');
        }
        $groupId = $_POST['groupId'];
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'input_error');
        }

        // privacy check
        $isOwner = OW::getUser()->getId() == $group->userId;
        $isModerator = OW::getUser()->isAuthorized('groups');
        $eventIisGroupsPlusManager = new OW_Event('iisgroupsplus.check.user.manager.status', array('groupId' => $group->id));
        OW::getEventManager()->trigger($eventIisGroupsPlusManager);
        if(isset($eventIisGroupsPlusManager->getData()['isUserManager'])){
            $isModerator=$eventIisGroupsPlusManager->getData()['isUserManager'];
        }
        if ( !$isOwner && !$isModerator )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $service = IISFILEMANAGER_BOL_Service::getInstance();
        $new_name = isset($_POST['new_name'])?$_POST['new_name']:null;
        $new_parent_id = isset($_POST['new_parent_id'])?$_POST['new_parent_id']:null;
        $service->editDirById($_POST['id'], $new_name, $new_parent_id);
        $subFolders = $service->getSubfolders('groups', (int) $group->id);
        return array('valid' => true, 'subfolders' => $subFolders);
    }

    public function deleteDir(){
        if ( !isset($_POST['groupId']) || !isset($_POST['id'])){
            return array('valid' => false, 'message' => 'input_error');
        }
        if( !IISSecurityProvider::checkPluginActive('iisfilemanager', true)) {
            return array('valid' => false, 'message' => 'input_error');
        }
        $groupId = $_POST['groupId'];
        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'input_error');
        }

        // privacy check
        $isOwner = OW::getUser()->getId() == $group->userId;
        $isModerator = OW::getUser()->isAuthorized('groups');
        $eventIisGroupsPlusManager = new OW_Event('iisgroupsplus.check.user.manager.status', array('groupId' => $group->id));
        OW::getEventManager()->trigger($eventIisGroupsPlusManager);
        if(isset($eventIisGroupsPlusManager->getData()['isUserManager'])){
            $isModerator=$eventIisGroupsPlusManager->getData()['isUserManager'];
        }
        if ( !$isOwner && !$isModerator )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $service = IISFILEMANAGER_BOL_Service::getInstance();
        $service->deleteDirById($_POST['id']);
        $subFolders = $service->getSubfolders('groups', (int) $group->id);
        return array('valid' => true, 'subfolders' => $subFolders);
    }

    public function canAddFile($group, $info = array()){
        if(!IISSecurityProvider::checkPluginActive('groups', true) || !IISSecurityProvider::checkPluginActive('iisgroupsplus', true)){
            return false;
        }

        if ( !OW::getUser()->isAuthenticated() || !OW::getUser()->isAuthorized('groups', 'view') )
        {
            return false;
        }

        if (!isset($info['checkUserExistInGroup']) || $info['checkUserExistInGroup']) {
            $isUserInGroup = GROUPS_BOL_Service::getInstance()->findUser($group->id, OW::getUser()->getId());
            if(!$isUserInGroup){
                return false;
            }
        }

        $canUploadFile = false;
        if (isset($info['groupSetting'])) {
            $groupSetting = $info['groupSetting'];
        } else {
            $groupSetting = IISGROUPSPLUS_BOL_GroupSettingDao::getInstance()->findByGroupId($group->id);
        }
        if (isset($groupSetting)){
            if($groupSetting->whoCanUploadFile == IISGROUPSPLUS_BOL_Service::WCU_MANAGERS)
            {
                $canUploadFile = true;
            }
        }

        if (isset($info['isChannel'])) {
            $isChannel = $info['isChannel'];
        } else {
            $isChannel = IISGROUPSPLUS_BOL_ChannelDao::getInstance()->findIsExistGroupId($group->id);
        }

        if (isset($info['isManager'])) {
            $isManager = $info['isManager'];
        } else {
            $isManager = IISGROUPSPLUS_BOL_GroupManagersDao::getInstance()->getGroupManagerByUidAndGid($group->id, OW::getUser()->getId());
        }

        $isCreator = $group->userId == OW::getUser()->getId() ? true : false;

        if ((isset($isManager) && $isManager) || $isCreator) {
            return true;
        }

        if (!OW::getUser()->isAuthorized('groups') && !OW::getUser()->isAdmin()) {
            return false;
        }

        if ($canUploadFile) {
            return true;
        }

        if (isset($isChannel)){
            return false;
        }

        return true;
    }

    public function canDeleteFile($group, $info = array()){
        if(!IISSecurityProvider::checkPluginActive('groups', true) || !IISSecurityProvider::checkPluginActive('iisgroupsplus', true)){
            return false;
        }

        if ( !OW::getUser()->isAuthenticated() || !OW::getUser()->isAuthorized('groups', 'view') )
        {
            return false;
        }

        if ( OW::getUser()->isAuthorized('groups') )
        {
            return true;
        }

        if (!isset($info['checkUserExistInGroup']) || $info['checkUserExistInGroup']) {
            $isUserInGroup = GROUPS_BOL_Service::getInstance()->findUser($group->id, OW::getUser()->getId());
            if (!$isUserInGroup) {
                return false;
            }
        }

        if (isset($info['isManager'])) {
            $isManager = $info['isManager'];
        } else {
            $isManager = IISGROUPSPLUS_BOL_GroupManagersDao::getInstance()->getGroupManagerByUidAndGid($group->id, OW::getUser()->getId());
        }

        $isCreator = $group->userId == OW::getUser()->getId() ? true : false;
        return ((isset($isManager) && $isManager) || $isCreator);
    }

    public function isFollowable($groupId, $groupDto = null, $info = array()) {
        if(!IISSecurityProvider::checkPluginActive('newsfeed', true)){
            return false;
        }

        if(!OW::getUser()->isAuthenticated()) {
            return false;
        }

        if ($groupId == null) {
            return false;
        }

        if ($groupDto == null) {
            $groupDto = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        }

        if ( $groupDto === null )
        {
            return false;
        }

        if ($groupDto->whoCanView == GROUPS_BOL_Service::WCV_INVITE) {
            if (!isset($info['checkUserExistInGroup']) || $info['checkUserExistInGroup']) {
                $userDtoInGroup = GROUPS_BOL_Service::getInstance()->findUser($groupId, OW::getUser()->getId());
                if ($userDtoInGroup == null) {
                    return false;
                }
            }
            return true;
        }

        return true;
    }

    public function isFollower($groupId) {
        if(!IISSecurityProvider::checkPluginActive('newsfeed', true)){
            return false;
        }
        if (!OW::getUser()->isAuthenticated()) {
            return false;
        }
        return NEWSFEED_BOL_Service::getInstance()->isFollow(OW::getUser()->getId(), 'groups', $groupId);
    }

    public function getFollowers($groupsId) {
        if(!IISSecurityProvider::checkPluginActive('newsfeed', true)){
            return array();
        }
        if (!OW::getUser()->isAuthenticated()) {
            return array();
        }
        return NEWSFEED_BOL_Service::getInstance()->isFollowByFeedIds(OW::getUser()->getId(), 'groups', $groupsId);
    }

    public function follow() {
        if(!IISSecurityProvider::checkPluginActive('newsfeed', true)){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        $groupId = null;
        if(isset($_GET['groupId'])){
            $groupId = (int) $_GET['groupId'];
        }

        if(!$this->isFollowable($groupId)) {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $eventParams = array(
            'userId' => OW::getUser()->getId(),
            'feedType' => 'groups',
            'feedId' => $groupId
        );
        OW::getEventManager()->call('feed.add_follow', $eventParams);
        return array('valid' => true, 'follow' => true, 'groupId' => $groupId);
    }

    public function unFollow() {
        if(!IISSecurityProvider::checkPluginActive('newsfeed', true)){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        $groupId = null;
        if(isset($_GET['groupId'])){
            $groupId = (int) $_GET['groupId'];
        }

        if(!$this->isFollowable($groupId)) {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $eventParams = array(
            'userId' => OW::getUser()->getId(),
            'feedType' => 'groups',
            'feedId' => $groupId
        );
        OW::getEventManager()->call('feed.remove_follow', $eventParams);
        return array('valid' => true, 'follow' => false, 'groupId' => $groupId);
    }

    public function isReplyFeatureEnable($entityType, $hasStatus = true,$groupId=null, $checkNewsfeedChatFormEvent = true) {
        $otpEventParams['includeWebService'] = true;
        if(isset($groupId))
        {
            $otpEventParams['groupId']= $groupId;
        }
        if ($checkNewsfeedChatFormEvent) {
            $otpEvent = OW_EventManager::getInstance()->trigger(new OW_Event('newsfeed.check.chat.form',$otpEventParams));
            if( !isset($otpEvent->getData()['canReply']) || !$otpEvent->getData()['canReply']){
                return false;
            }
        } else {
            $addReplyFeatureConfig = OW::getConfig()->getValue('newsfeed', 'addReply');
            if (!isset($addReplyFeatureConfig) || $addReplyFeatureConfig != "on") {
                return false;
            }
        }
        if (!in_array($entityType, ['groups-status','user-status']) || !$hasStatus) {
            return false;
        }
        return true;
    }

    public function getGroupsInformation($groups, $first = 0, $count = 10, $params = array('users', 'files', 'posts', 'subGroups'), $additionalInfo = array()) {
        $info = array();
        $groupIds = array();

        foreach ($groups as $group) {
            $groupIds[] = $group->id;
        }

        if(IISSecurityProvider::checkPluginActive('iisgroupsplus', true)) {
            $additionalInfo['groups_category_information'] = IISGROUPSPLUS_BOL_Service::getInstance()->getGroupCategoryByGroupIds($groupIds);
            $additionalInfo['groups_channel_ids'] = IISGROUPSPLUS_BOL_ChannelDao::getInstance()->findIsExistGroupIds($groupIds);
            $additionalInfo['groups_settings'] = IISGROUPSPLUS_BOL_GroupSettingDao::getInstance()->findByGroupIds($groupIds);
            $additionalInfo['groups_manager_ids'] = IISGROUPSPLUS_BOL_GroupManagersDao::getInstance()->getGroupManagersByGroupIds($groupIds);
            $additionalInfo['current_user_groups_follow'] = $this->getFollowers($groupIds);
            $additionalInfo['groups_last_action_seen'] = GROUPS_BOL_GroupUserDao::getInstance()->findByGroupsAndUserId($groupIds, OW::getUser()->getId());
        }

        foreach ($groups as $group) {
            $info[] = $this->getGroupInformation($group, $first, $count, $params, $additionalInfo);
        }
        return $info;
    }

    public function getGroupInformation($group, $first = 0, $count = 10, $params = array('users', 'files', 'posts', 'subGroups'), $additionalInfo = array()){
        $imageUrl = GROUPS_BOL_Service::getInstance()->getGroupImageUrl($group);
        $imageUrlPath = GROUPS_BOL_Service::getInstance()->getGroupImagePath($group, GROUPS_BOL_Service::IMAGE_SIZE_SMALL);
        $emptyImage = empty($imageUrlPath) ? true : false;
        $description = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->stripString($group->description, false);

        $checkCanView = true;
        if (isset($additionalInfo['checkCanView'])) {
            $checkCanView = $additionalInfo['checkCanView'];
        }
        $canView = true;
        if ($checkCanView) {
            $canView = GROUPS_BOL_Service::getInstance()->isCurrentUserCanView($group);
        }

        $groupOwnerInfo = IISMOBILESUPPORT_BOL_WebServiceUser::getInstance()->getUserInformationById($group->userId);
        if (!$canView)
        {
            $invite = GROUPS_BOL_Service::getInstance()->findInvite($group->id, OW::getUser()->getId());
            if ($invite != null){
                $result = array(
                    "id" => (int) $group->id,
                    "invite" => true,
                    "title" => $group->title,
                    "description" => $description,
                    "privacy" => $group->privacy,
                    "userId" => (int) $group->userId,
                    "user" => $groupOwnerInfo,
                    'emptyImage' => $emptyImage,
                    'imageInfo' => BOL_AvatarService::getInstance()->getAvatarInfo((int) $group->id, $imageUrl, 'group'),
                    "timestamp" => $group->timeStamp,
                    "imageUrl" => $imageUrl,
                    "whoCanView" => $group->whoCanView,
                    "whoCanInvite" => $group->whoCanInvite,
                    "status" =>$group->status,
                    "followable" => false,
                    "follower" => false,
                    "whoCanUploadFile" => 'manager',
                    "whoCanCreateTopic" => 'manager',
                );

                $eventFindParent = OW::getEventManager()->trigger(new OW_Event('iissubgroups.groups_find_parent', array('groupId'=> $group->id) ));
                if(isset($eventFindParent->getData()['parentId'])){
                    $parentId = $eventFindParent->getData()['parentId'];
                    $parentObject = GROUPS_BOL_Service::getInstance()->findGroupById($parentId);
                    if (isset($parentObject)) {
                        $result['parent'] = array(
                            'title' => $parentObject->title,
                            'id' => (int) $parentId,
                        );
                    }
                }

                return $result;
            }
            return array();
        }

        $categoryValue = "";
        $registered = false;
        $whoCanCreateContent = 'group';
        $whoCanUploadFile = 'participant';
        $whoCanCreateTopic = 'participant';
        $usersCount = 0;

        $filesInformation = array();
        $managerIds = array();
        if (isset($additionalInfo['groups_category_information'])) {
            if (isset($additionalInfo['groups_category_information'][$group->id])) {
                $categoryValue = $additionalInfo['groups_category_information'][$group->id]['label'];
            }
        } else {
            if(IISSecurityProvider::checkPluginActive('iisgroupsplus', true)) {
                $categoryId = IISGROUPSPLUS_BOL_Service::getInstance()->getGroupCategoryByGroupId($group->id);
                if ($categoryId != null) {
                    $category = IISGROUPSPLUS_BOL_Service::getInstance()->getCategoryById($categoryId);
                    if ($category != null) {
                        $categoryValue = $category->label;
                    }
                }
            }
        }

        if (isset($additionalInfo['groups_channel_ids'])) {
            if (in_array($group->id, $additionalInfo['groups_channel_ids'])) {
                $whoCanCreateContent = 'channel';
            }
        } else {
            if(IISSecurityProvider::checkPluginActive('iisgroupsplus', true)) {
                $channel = IISGROUPSPLUS_BOL_ChannelDao::getInstance()->findIsExistGroupId($group->id);
                if (isset($channel)) {
                    $whoCanCreateContent = IISGROUPSPLUS_BOL_Service::WCC_CHANNEL;
                }
            }
        }

        $groupSetting = null;
        if (isset($additionalInfo['groups_settings'])) {
            if (isset($additionalInfo['groups_settings'][$group->id])) {
                $groupSetting = $additionalInfo['groups_settings'][$group->id];
                $whoCanCreateTopic = $groupSetting->whoCanCreateTopic;
                $whoCanUploadFile = $groupSetting->whoCanUploadFile;
            }
        } else {
            if(IISSecurityProvider::checkPluginActive('iisgroupsplus', true)) {
                $groupSetting = IISGROUPSPLUS_BOL_GroupSettingDao::getInstance()->findByGroupId($group->id);
                if (isset($groupSetting)) {
                    $whoCanUploadFile = $groupSetting->getWhoCanUploadFile();
                    $whoCanCreateTopic = $groupSetting->getWhoCanCreateTopic();
                }
            }
        }

        if (isset($additionalInfo['groups_manager_ids'])) {
            if (isset($additionalInfo['groups_manager_ids'][$group->id])) {
                $managerIds = $additionalInfo['groups_manager_ids'][$group->id];
            }
        } else {
            if(IISSecurityProvider::checkPluginActive('iisgroupsplus', true)) {
                $managers = IISGROUPSPLUS_BOL_GroupManagersDao::getInstance()->getGroupManagersByGroupId($group->id);
                foreach ($managers as $manager) {
                    $managerIds[] = (int)$manager->userId;
                }
            }
        }

        if(IISSecurityProvider::checkPluginActive('iisgroupsplus', true)){
            if (in_array('files', $params)) {
                $filesList = IISGROUPSPLUS_BOL_Service::getInstance()->findFileList($group->id, $first, $count);
                $filesInformation = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->preparedFileList($group, $filesList);
            }
        }

        $users = array();
        $pendingUsers = array();
        $pendingUsersId = array();
        if (in_array('users', $params)) {
            $idList = array();
            $usersCount = GROUPS_BOL_Service::getInstance()->findUserListCount($group->id);
            $groupUserList = GROUPS_BOL_GroupUserDao::getInstance()->findListByGroupId($group->id, $first, $count);
            foreach ($groupUserList as $groupUser) {
                $idList[] = $groupUser->userId;
            }

            if (IISSecurityProvider::checkPluginActive('iisgroupsplus', true)){
                $pendingUsersList = GROUPS_BOL_Service::getInstance()->findAllInviteList($group->id);
                foreach ($pendingUsersList as $pendingUser) {
                    $pendingUsersId[] = $pendingUser->userId;
                    $idList[] = $pendingUser->userId;
                }
            }


            $idList = array_unique($idList);
            $usersObject = BOL_UserService::getInstance()->findUserListByIdList($idList);
            $usernames = BOL_UserService::getInstance()->getDisplayNamesForList($idList);
            $avatars = BOL_AvatarService::getInstance()->getAvatarsUrlList($idList);
            foreach ($usersObject as $userObject) {
                $username = null;
                if (isset($usernames[$userObject->id])) {
                    $username = $usernames[$userObject->id];
                }

                $avatarUrl = null;
                if (isset($avatars[$userObject->id])) {
                    $avatarUrl = $avatars[$userObject->id];
                }
                $userData = IISMOBILESUPPORT_BOL_WebServiceUser::getInstance()->populateUserData($userObject, $avatarUrl, $username);
                $userData['isManager'] = false;
                if (in_array($userObject->id, $managerIds)){
                    $userData['isManager'] = true;
                }
                if (!in_array($userObject->id, $pendingUsersId)) {
                    $users[] = $userData;
                } else {
                    $pendingUsers[] = $userData;
                }
            }
        }

        $canInviteUser = true;
        $isCurrentUserManager = in_array(OW::getUser()->getId(), $managerIds);
        $checkUserExistInGroup = true;
        if (isset($additionalInfo['checkUserExistInGroup'])) {
            $checkUserExistInGroup = $additionalInfo['checkUserExistInGroup'];
        }
        if (!$isCurrentUserManager && !GROUPS_BOL_Service::getInstance()->isCurrentUserInviteByGroupObject($group, false, $checkUserExistInGroup)) {
            $canInviteUser = false;
        }

        $canCreatePost = true;

        if(OW::getUser()->isAuthenticated()){
            if ($checkUserExistInGroup) {
                $registeredUser = GROUPS_BOL_Service::getInstance()->findUser($group->id, OW::getUser()->getId());
                if($registeredUser != null){
                    $registered = true;
                }
            } else {
                $registered = true;
            }

            if ($whoCanCreateContent == 'channel' && !$isCurrentUserManager) {
                $channelEvent = OW::getEventManager()->trigger(new OW_Event('iisgroupsplus.on.channel.add.widget',
                    array('feedId'=> $group->id, 'feedType'=> 'groups', 'isChannel' => $whoCanCreateContent == 'channel', 'isManager' => $isCurrentUserManager, 'group' => $group ) ));
                if(isset($channelEvent->getData()['channelParticipant'])){
                    $channelEvent->getData()['channelParticipant'];
                    $isChannelParticipant = $channelEvent->getData()['channelParticipant'];
                    if($isChannelParticipant){
                        $canCreatePost = false;
                    }
                }
            }

        }

        $registrable = false;
        if(!$registered && ($group->whoCanView != 'invite' || OW::getUser()->isAdmin())) {
            $registrable = true;
        }

        $isAdmin = $isCurrentUserManager || GROUPS_BOL_Service::getInstance()->isCurrentUserCanEdit($group, false);
        $groupInfo = array(
            'checkUserExistInGroup' => $checkUserExistInGroup,
            'isChannel' => $whoCanCreateContent == 'channel',
            'isManager' => $isCurrentUserManager,
            'groupSetting' => $groupSetting,
            'canCreatePost' => $canCreatePost,
            'group_object' => $group,
        );
        if (isset($additionalInfo['doPrepareActions'])) {
            $groupInfo['doPrepareActions'] = $additionalInfo['doPrepareActions'];
        }
        $canAddFile = $this->canAddFile($group, $groupInfo);
        $canDeleteFolder = $canDeleteFile = $this->canDeleteFile($group, $groupInfo);
        $canAddTopic = $this->canAddTopicToGroup($group->id, $groupInfo);

        $lastSeen = null;
        if (isset($_GET['lastSeenPostTime']) && $_GET['lastSeenPostTime'] != null && $first == 0) {
            $lastSeen = $_GET['lastSeenPostTime'];
        }

        $unreadCount = GROUPS_BOL_Service::getInstance()->getUnreadCountForGroupUser($group->id, false, $lastSeen);
        $canAddSubGroups = false;
        $eventHasAccess = OW::getEventManager()->trigger(new OW_Event('iissubgroup.check.access.create.subgroups', array('groupId' => $group->id)));
        if (isset($eventHasAccess->getData()['canCreateSubGroup']) && $eventHasAccess->getData()['canCreateSubGroup']) {
            $canAddSubGroups = true;
        }

        $postCount = max($count, $unreadCount);
        $postCount = min($postCount, IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->getCachedGroupPostsSize());
        $postsInformation = $this->getGroupPosts($group->id, $first, $postCount, $groupInfo);

        $lastActivityString = OW::getLanguage()->text('groups','feed_create_string');
        $lastActivityUsername = $groupOwnerInfo['name'];
        $lastActivityTimestamp = $group->timeStamp;
        if($postsInformation != null && sizeof($postsInformation) > 0){
            $find = false;
            foreach ($postsInformation as $postInformation){
                if($find){
                    break;
                }
                $lastActivity = $postInformation;
                if(isset($lastActivity['text']) && !empty($lastActivity['text'])){
                    $find = true;
                    $lastActivityString = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->stripString($lastActivity['text'], true, true);
                    if(isset($lastActivity['user']['name'])){
                        $lastActivityUsername = $lastActivity['user']['name'];
                    }

                    if(isset($lastActivity['time'])){
                        $lastActivityTimestamp = $lastActivity['time'];
                    }
                } else if($lastActivity['entityType'] == 'groups-add-file'){
                    $find = true;
                    $lastActivityString = OW::getLanguage()->text('iismobilesupport', 'add_file_string');
                    if(isset($lastActivity['user']['name'])){
                        $lastActivityUsername = $lastActivity['user']['name'];
                    }

                    if(isset($lastActivity['time'])){
                        $lastActivityTimestamp = $lastActivity['time'];
                    }
                } else if(in_array($lastActivity['entityType'], array('groups-join', 'groups-leave')) && !empty($lastActivity['activityString'])){
                    $find = true;
                    $lastActivityString = $lastActivity['activityString'];
                    if(isset($lastActivity['user']['name'])){
                        $lastActivityUsername = $lastActivity['user']['name'];
                    }

                    if(isset($lastActivity['time'])){
                        $lastActivityTimestamp = $lastActivity['time'];
                    }
                } else if(in_array($lastActivity['entityType'], array('forum-topic', 'forum-post')) && !empty($lastActivity['activityString'])){
                    $find = true;
                    $lastActivityString = $lastActivity['activityString'];
                    if(isset($lastActivity['user']['name'])){
                        $lastActivityUsername = $lastActivity['user']['name'];
                    }

                    if(isset($lastActivity['time'])){
                        $lastActivityTimestamp = $lastActivity['time'];
                    }
                }
            }
        }

        $followable = !$checkUserExistInGroup || $this->isFollowable($group->id, $group, $groupInfo);
        $isFollower = false;
        if (isset($additionalInfo['current_user_groups_follow'])) {
            if (isset($additionalInfo['current_user_groups_follow'][$group->id])) {
                $isFollower = true;
            }
        } else {
            $this->isFollower($group->id);
        }

        $data = array(
            "id" => (int) $group->id,
            "title" => $group->title,
            "description" => $description,
            "privacy" => $group->privacy,
            "unreadCount" => (int) $unreadCount,
            "can_add_file" => $canAddFile,
            "can_add_sub_groups" => $canAddSubGroups,
            "pendingUsers" => $pendingUsers,
            "can_add_topic" => $canAddTopic,
            "can_delete_file" => $canDeleteFile,
            "userId" => (int) $group->userId,
            "user" => $groupOwnerInfo,
            "timestamp" => $group->timeStamp,
            "imageUrl" => $imageUrl,
            'imageInfo' => BOL_AvatarService::getInstance()->getAvatarInfo((int) $group->id, $imageUrl, 'group'),
            'emptyImage' => $emptyImage,
            "categoryValue" => $categoryValue,
            "isAdmin" => $isAdmin,
            "registered" => $registered,
            "registrable" => $registrable,
            "files" => $filesInformation,
            "can_create_post" => $canCreatePost,
            "can_invite_user" => $canInviteUser,
            "users_count" => $usersCount,
            "whoCanView" => $group->whoCanView,
            "whoCanInvite" => $group->whoCanInvite,
            "whoCanCreateContent" => $whoCanCreateContent,
            "whoCanUploadFile" => $whoCanUploadFile,
            "whoCanCreateTopic" => $whoCanCreateTopic,
            "lastActivityString" => $lastActivityString,
            "lastActivityUsername" => $lastActivityUsername,
            "lastActivityTimestamp" => $lastActivityTimestamp,
            "users" => $users,
            "followable" => $followable,
            "follower" => $isFollower,
            "status" => $group->status
        );

        if(IISSecurityProvider::checkPluginActive('iisfilemanager', true)){
            if (in_array('files', $params)) {
                $data['subfolders'] = IISFILEMANAGER_BOL_Service::getInstance()->getSubfolders('groups', (int) $group->id);
                $data['can_delete_folder'] = $canDeleteFolder;
            }
        }

        if (isset($additionalInfo['groups_last_action_seen'][$group->id]->last_seen_action)) {
            $data['lastSeenPostTime'] = (int) $additionalInfo['groups_last_action_seen'][$group->id]->last_seen_action;
        } else {
            $groupUser = GROUPS_BOL_GroupUserDao::getInstance()->findGroupUser($group->id, OW::getUser()->getId());
            if ($groupUser != null) {
                $data['lastSeenPostTime'] = (int) $groupUser->last_seen_action;
            }
        }

        $eventFindParent = OW::getEventManager()->trigger(new OW_Event('iissubgroups.groups_find_parent', array('groupId'=> $group->id) ));
        if(isset($eventFindParent->getData()['parentId'])){
            $parentId = $eventFindParent->getData()['parentId'];
            $parentObject = GROUPS_BOL_Service::getInstance()->findGroupById($parentId);
            if (isset($parentObject)) {
                $data['parent'] = array(
                    'title' => $parentObject->title,
                    'id' => (int) $parentId,
                );
            }
        }

        if (in_array('posts', $params)) {
            $data["posts"] = $postsInformation;
            $postEntityIds = array();
            foreach ($postsInformation as $postInfo) {
                if (isset($postInfo['entityId'])) {
                    $postEntityIds[] = $postInfo['entityId'];
                }
            }
            if (sizeof($postEntityIds) > 0 && IISSecurityProvider::checkPluginActive('notifications', true)) {
                $unMarkedNotifications = array();
                $groupPostsNotifications = NOTIFICATIONS_BOL_NotificationDao::getInstance()->findNotificationsByEntityIds('groups-status', $postEntityIds, OW::getUser()->getId());
                if($groupPostsNotifications != null && is_array($groupPostsNotifications)) {
                    foreach ($groupPostsNotifications as $cNotif) {
                        if ($cNotif->viewed != 1) {
                            $unMarkedNotifications[] = $cNotif->id;
                        }
                    }
                }

                $groupFilesNotifications = NOTIFICATIONS_BOL_NotificationDao::getInstance()->findNotificationsByEntityIds('groups-add-file', $postEntityIds, OW::getUser()->getId());
                if($groupFilesNotifications != null && is_array($groupFilesNotifications)) {
                    foreach ($groupFilesNotifications as $cNotif) {
                        if ($cNotif->viewed != 1) {
                            $unMarkedNotifications[] = $cNotif->id;
                        }
                    }
                }

                $groupTopicsNotifications = NOTIFICATIONS_BOL_NotificationDao::getInstance()->findNotificationsByEntityIds('group-topic-add', $postEntityIds, OW::getUser()->getId());
                if($groupTopicsNotifications != null && is_array($groupTopicsNotifications)) {
                    foreach ($groupTopicsNotifications as $cNotif) {
                        if ($cNotif->viewed != 1) {
                            $unMarkedNotifications[] = $cNotif->id;
                        }
                    }
                }

                $groupJoinsNotifications = NOTIFICATIONS_BOL_NotificationDao::getInstance()->findNotificationsByEntityIds('groups-join', $postEntityIds, OW::getUser()->getId());
                if($groupJoinsNotifications != null && is_array($groupJoinsNotifications)) {
                    foreach ($groupJoinsNotifications as $cNotif) {
                        if ($cNotif->viewed != 1) {
                            $unMarkedNotifications[] = $cNotif->id;
                        }
                    }
                }

                if (sizeof($unMarkedNotifications) > 0){
                    NOTIFICATIONS_BOL_NotificationDao::getInstance()->markViewedByIds($unMarkedNotifications);
                }
            }
        }

        if (in_array('subGroups', $params)) {
            $subGroups = array();

            $eventGetSubGroups = new OW_Event('groups_find_subgroups', array('groupId' => $group->id, 'first' => $first, 'count' => $count));
            OW::getEventManager()->trigger($eventGetSubGroups);
            if(isset($eventGetSubGroups->getData()['groups'])){
                $subGroups = $eventGetSubGroups->getData()['groups'];
            }

            $subGroupsData = $this->getGroupsInformation($subGroups, 0, 2, array());

            if (sizeof($subGroupsData) > 0) {
                $data["subGroups"] = $subGroupsData;
            }
        }

        return $data;
    }


    private function getGroupPosts($groupId, $first = 0, $count = 11, $additionalInfo = array()){
        if ($count != 11) {
            $count += 1;
        }
        $params = array(
            "feedType" => "groups",
            "feedId" => $groupId,
            "offset" => $first,
            "displayCount" => $count,
            "displayType" => "action",
            "checkMore" => true,
            "feedAutoId" => "feed1",
            "startTime" => time(),
            "formats" => null,
            "endTIme" => 0,
            "additionalInfo" => $additionalInfo,
        );
        return IISMOBILESUPPORT_BOL_WebServiceNewsfeed::getInstance()->getActionData($params);
    }

    public function getGroupFields(){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);

        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        $fields = array();
        $language = OW::getLanguage();
        $fields[] = array(
            'name' => 'title',
            'type' => 'text',
            'label' => $language->text('groups', 'create_field_title_label'),
            'presentation' => 'text',
            'values' => array()
        );

        $fields[] = array(
            'name' => 'description',
            'type' => 'text',
            'label' => $language->text('groups', 'create_field_description_label'),
            'presentation' => 'text',
            'values' => array()
        );

        $whoCanViewValues[$language->text('groups', 'form_who_can_view_anybody')] = GROUPS_BOL_Service::WCV_ANYONE;
        $whoCanViewValues[$language->text('groups', 'form_who_can_view_invite')] = GROUPS_BOL_Service::WCV_INVITE;
        $fields[] = array(
            'name' => 'whoCanView',
            'type' => 'select',
            'label' => $language->text('groups', 'form_who_can_view_label'),
            'presentation' => 'radio',
            'values' => $whoCanViewValues
        );

        $whoCanInviteValues[$language->text('groups', 'form_who_can_invite_participants')] = GROUPS_BOL_Service::WCI_PARTICIPANT;
        $whoCanInviteValues[$language->text('groups', 'form_who_can_invite_creator')] = GROUPS_BOL_Service::WCI_CREATOR;

        $fields[] = array(
            'name' => 'whoCanInvite',
            'type' => 'select',
            'label' => $language->text('groups', 'form_who_can_invite_label'),
            'presentation' => 'radio',
            'values' => $whoCanInviteValues
        );

        if(IISSecurityProvider::checkPluginActive('iisgroupsplus', true)) {
            $whoCanCreateContentValues[$language->text('iisgroupsplus', 'form_who_can_create_content_participants')] = IISGROUPSPLUS_BOL_Service::WCC_GROUP;
            $whoCanCreateContentValues[$language->text('iisgroupsplus', 'form_who_can_create_content_creators')] = IISGROUPSPLUS_BOL_Service::WCC_CHANNEL;

            $fields[] = array(
                'name' => 'whoCanCreateContent',
                'type' => 'select',
                'label' => $language->text('iisgroupsplus', 'who_can_create_content'),
                'presentation' => 'radio',
                'values' => $whoCanCreateContentValues,
                'required' => false
            );

            $categories = IISGROUPSPLUS_BOL_Service::getInstance()->getGroupCategoryList();
            if(sizeof($categories) > 0) {
                $values = array();
                $values[null] = OW::getLanguage()->text('iisgroupsplus', 'select_category');
                foreach ($categories as $category) {
                    $values[$category->label] = $category->id;
                }
                $fields[] = array(
                    'name' => 'categoryStatus',
                    'type' => 'select',
                    'label' => $language->text('iisgroupsplus', 'select_category'),
                    'presentation' => 'radio',
                    'values' => $values,
                    'required' => false
                );
            }
        }

        return $fields;
    }

    public function processCreateGroup(){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);

        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        if ( !OW::getUser()->isAuthenticated() || !OW::getUser()->isAuthorized('groups', 'create') )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $valid = true;
        $questions = $this->getGroupFields();
        $formValidator = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->checkDataFormValid($questions);
        if($formValidator['valid'] == true){
            $result = $this->createGroup();
            if($result == null){
                $valid = false;
            }
            if($valid) {
                return array(
                    'valid' => true,
                    'message' => 'group_created',
                    'group' => $this->getGroupInformation($result, 0, 2, array()),
                );
            }
        }else{
            $valid = false;
        }

        if(!$valid){
            return array('valid' => false, 'message' => 'invalid_data');
        }
    }

    public function processEditGroup(){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);

        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        if ( !OW::getUser()->isAuthenticated() || !OW::getUser()->isAuthorized('groups', 'create') )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $groupId = null;
        if(isset($_GET['groupId'])){
            $groupId = (int) $_GET['groupId'];
        }

        if($groupId == null){
            return array('valid' => false, 'message' => 'input_error');
        }

        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $isAdmin = GROUPS_BOL_Service::getInstance()->isCurrentUserCanEdit($group);
        if(!$isAdmin){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $valid = true;
        $questions = $this->getGroupFields();
        $formValidator = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->checkDataFormValid($questions);
        if($formValidator['valid'] == true){
            if(!isset($_POST['whoCanCreateContent'])){
                $_POST['whoCanCreateContent'] = 'group';
            }
            $data = $_POST;
            if (isset($_FILES['file'])){
                if ( !empty($_FILES['file']['name']) ){
                    if ( (int) $_FILES['file']['error'] !== 0 ||
                        !is_uploaded_file($_FILES['file']['tmp_name']) ||
                        !UTIL_File::validateImage($_FILES['file']['name']) ){
                        // Do nothing
                    }
                    else{
                        $isFileClean = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->isFileClean($_FILES['file']['tmp_name']);
                        if ($isFileClean) {
                            $data['image'] = $_FILES['file'];
                        }
                    }
                }
            }
            $result = GROUPS_BOL_Service::getInstance()->processGroupInfo($group, $data);
            if($result == null){
                $valid = false;
            }
            if($valid) {
                return array(
                    'valid' => true,
                    'message' => 'group_edited',
                    'group' => array(
                        'id' => (int) $result->id,
                        'time' => $result->timeStamp,
                        'userId' => (int) $result->userId,
                        'title' => $result->title,
                        'description' => $result->description,
                        'whoCanInvite' => $result->whoCanInvite,
                        'whoCanView' => $result->whoCanView,
                    ));
            }
        }else{
            $valid = false;
        }

        if(!$valid){
            return array('valid' => false, 'message' => 'invalid_data');
        }
    }

    public function joinGroup(){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);

        if(!$pluginActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        $guestAccess = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->checkGuestAccess();
        if(!$guestAccess){
            return array('valid' => false, 'message' => 'guest_cant_view');
        }

        if ( !OW::getUser()->isAuthorized('groups', 'view') )
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $groupId = null;
        $userId = OW::getUser()->getId();
        if(isset($_GET['groupId'])){
            $groupId = (int) $_GET['groupId'];
        }

        if($groupId == null){
            return array('valid' => false, 'message' => 'authorization_error');
        }
        $findUser = GROUPS_BOL_Service::getInstance()->findUser($groupId, $userId);
        if($findUser != null){
            return array('valid' => true, 'message' => 'add_before');
        }

        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        $invite = GROUPS_BOL_Service::getInstance()->findInvite($groupId, $userId);

        if ( $invite !== null )
        {
            GROUPS_BOL_Service::getInstance()->markInviteAsViewed($groupId, $userId);
        }
        else if ( $group->whoCanView == GROUPS_BOL_Service::WCV_INVITE  && !OW::getUser()->isAdmin() && !OW::getUser()->isAuthorized('groups'))
        {
            return array('valid' => false, 'message' => 'authorization_error');
        }

        GROUPS_BOL_Service::getInstance()->addUser($groupId, $userId);
        $groupData = $this->getGroupInformation($group);

        return array('valid' => true, 'message' => 'user_add', 'group' => $groupData);
    }

    public function createGroup(){
        if(!OW::getUser()->isAuthenticated()){
            return null;
        }

        if(!in_array($_POST['whoCanInvite'], array(GROUPS_BOL_Service::WCI_CREATOR, GROUPS_BOL_Service::WCI_PARTICIPANT))) {
            return null;
        }

        if(!in_array($_POST['whoCanView'], array(GROUPS_BOL_Service::WCV_ANYONE, GROUPS_BOL_Service::WCV_INVITE))) {
            return null;
        }

        $data = $_POST;
        if (isset($_FILES['file'])){
            if ( !empty($_FILES['file']['name']) ){
                if ( (int) $_FILES['file']['error'] !== 0 ||
                    !is_uploaded_file($_FILES['file']['tmp_name']) ||
                    !UTIL_File::validateImage($_FILES['file']['name']) ){
                    // Do nothing
                }
                else{
                    $isFileClean = IISMOBILESUPPORT_BOL_WebServiceGeneral::getInstance()->isFileClean($_FILES['file']['tmp_name']);
                    if ($isFileClean) {
                        $data['image'] = $_FILES['file'];
                    }
                }
            }
        }
        $group = GROUPS_BOL_Service::getInstance()->createGroup(OW::getUser()->getId(), $data);
        return $group;
    }

    public function activateGroup(){
        $pluginActive = IISSecurityProvider::checkPluginActive('groups', true);
        $pluginPlusActive = IISSecurityProvider::checkPluginActive('iisgroupsplus', true);

        if(!$pluginActive || !$pluginPlusActive){
            return array('valid' => false, 'message' => 'plugin_not_found');
        }

        $groupId = null;
        if(isset($_GET['groupId'])){
            $groupId = (int) $_GET['groupId'];
        }

        if($groupId == null){
            return array('valid' => false, 'message' => 'input_error');
        }

        $group = GROUPS_BOL_Service::getInstance()->findGroupById($groupId);
        if($group == null){
            return array('valid' => false, 'message' => 'authorization_error');
        }

        if(IISGROUPSPLUS_BOL_Service::getInstance()->approveGroupById($groupId))
        {
            return array('valid' => true, 'message' => 'group_activated', 'group' => $group);
        }
        else
        {
            return array('valid' => false, 'message' => 'group_activation_failed', 'group' => $group);
        }
    }
}