<?php
/**
 * Config service.
 * 
 * @package ow_system_plugins.base.bol
 * @since 1.0
 */
final class BOL_ConfigService
{
    const EVENT_BEFORE_SAVE = "base.before_config_save";
    const EVENT_AFTER_SAVE = "base.after_config_save";
    
    const EVENT_BEFORE_REMOVE = "base.before_config_remove";
    const EVENT_AFTER_REMOVE = "base.after_config_remove";
    
    /**
     * @var BOL_ConfigDao
     */
    private $configDao;
    /**
     * @var BOL_ConfigService
     */
    private static $classInstance;

    /**
     * Returns an instance of class (singleton pattern implementation).
     *
     * @return BOL_ConfigService
     */
    public static function getInstance()
    {
        if ( self::$classInstance === null )
        {
            self::$classInstance = new self();
        }

        return self::$classInstance;
    }

    /**
     * Constructor.
     */
    private function __construct()
    {
        $this->configDao = BOL_ConfigDao::getInstance();
    }

    /**
     * Returns config value for provided key and name.
     *
     * @param string $key
     * @param string $name
     * @return string
     */
    public function findConfigValue( $key, $name )
    {
        $config = $this->configDao->findConfig($key, $name);

        if ( $config === null )
        {
            return null;
        }

        return $config->getValue();
    }

    /**
     * Returns config item for provided key and name.
     * 
     * @param $key
     * @param $name
     * @return unknown_type
     */
    public function findConfig( $key, $name )
    {
        return $this->configDao->findConfig($key, $name);
    }

    /**
     * Returns config items list for provided plugin key.
     * 
     * @param string $key
     * @return array
     */
    public function findConfigsList( $key )
    {
        return $this->configDao->findConfigsList($key);
    }

    /**
     * Returns all configs.
     *
     * @return array<BOL_Config>
     */
    public function findAllConfigs()
    {
        return $this->configDao->findAll();
    }

    /**
     * Adds new config item.
     * 
     * @param string $key
     * @param string $name
     * @param mixed $value
     */
    public function addConfig( $key, $name, $value, $description = null )
    {
        if ( $this->findConfig($key, $name) !== null )
        {
            throw new InvalidArgumentException("Can't add config `" . $name . "` in section `" . $key . "`. Duplicated key and name!");
        }

        $newConfig = new BOL_Config();
        $newConfig->setKey($key)->setName($name)->setValue($value)->setDescription($description);
        
        $event = OW::getEventManager()->trigger(new OW_Event(self::EVENT_BEFORE_SAVE, array(
            "key" => $key,
            "name" => $name,
            "value" => $value,
            "oldValue" => null
        ), $value));
        
        $newConfig->setValue($event->getData());
        
        $this->configDao->save($newConfig);
    }

    /**
     * Updates config item value.
     * 
     * @param string $key
     * @param string $name
     * @param mixed $value
     * @param $description
     * @throws InvalidArgumentException
     */
    public function saveConfig( $key, $name, $value, $description=null )
    {
        $config = $this->configDao->findConfig($key, $name);

        if ($config === null) {
            $this->addConfig($key, $name, $value, $description);
        } else {
            $event = OW::getEventManager()->trigger(new OW_Event(self::EVENT_BEFORE_SAVE, array(
                "key" => $key,
                "name" => $name,
                "value" => $value,
                "oldValue" => $config->getValue()
            ), $value));
            $config->setValue($event->getData());

            if(isset($description)){
                $config->setDescription($description);
            }

            $this->configDao->save($config);

            OW::getEventManager()->trigger(new OW_Event(self::EVENT_AFTER_SAVE, array(
                "key" => $key,
                "name" => $name,
                "value" => $value,
                "oldValue" => $config->getValue()
            )));
        }
    }

    /**
     * Removes config item by provided plugin key and config name.
     * 
     * @param string $key
     * @param string $name
     */
    public function removeConfig( $key, $name )
    {
        $event = OW::getEventManager()->trigger(new OW_Event(self::EVENT_BEFORE_REMOVE, array(
            "key" => $key,
            "name" => $name
        )));
        
        if ( $event->getData() !== false )
        {
            $this->configDao->removeConfig($key, $name);
            
            OW::getEventManager()->trigger(new OW_Event(self::EVENT_BEFORE_REMOVE, array(
                "key" => $key,
                "name" => $name
            )));
        }
    }

    /**
     * Removes all plugin configs.
     * 
     * @param string $pluginKey
     */
    public function removePluginConfigs( $pluginKey )
    {
        $this->configDao->removeConfigs($pluginKey);
    }
}